function SMIDIBT_soundmod_latencies(data_dir)
% SMIDIT_soundmod_latencies(data_dir)
%   Example script for extracting MIDI and audio onset times using the
%   Schultz MIDI Benchmarking Toolbox to get the latencies of all files 
%   within a folder
%
%   INPUT:
%   data_dir        The name of a folder containing the ".wav" files for
%                   the MIDI triggers (ending in "_trigs") and also
%                   contains the complementary audio files (ending in 
%                   "_audio") to extract the audio onsets. The default is
%                   the folder "soundmod_data", which was provided in the 
%                   downloaded file.
%
%   Dependencies can be found in the "MATLAB" folder of the SMIDIB Toolbox
%   hosted here: www.band-lab.com/smidi_toolbox
%
%   2017-09-28 ben.schultz@maastrichtuniversity.nl
%   Copyright (c) 2017, Benjamin Schultz, Maastricht University.

%   This script is described in more detail in the publication:
%   Schultz, B. G. (submitted).The Schultz MIDI Benchmarking Toolbox for
%   MIDI interfaces, percussion pads, and sound cards. Behavior Research 
%   Methods.
%
%   The SMIDIB Toolbox is distributed in the hope that it will be useful but
%   WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%   General Public License for more details.
% 
%   You should have received a copy of the GNU General Public License
%   along with the SMIDIB Toolbox; if not, write to the Free Software
%   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
%   02110-1301 USA
% 
%   See the file "COPYING" for the text of the license.

%% Preset necessary information
% set directory for .wav files
if ~exist('data_dir','var')
    data_dir = fullfile(cd,'soundmod_data');
end

% set the parameters (if not the default)
params.plot_dir = fullfile(cd,'soundmod_plots');
params.out_dir = data_dir;

% find all .wav files
all_filenames = dir(fullfile(data_dir,'*.wav'));
all_filenames = {all_filenames.name};

% select only the trigs
file_ind = strfind(all_filenames,'_trigs');
all_filenames = all_filenames(cellfun(@isempty,file_ind)==0);
num_files = length(all_filenames);

% preset output headers
headers = {'ID','Device','Instrument',...
    'Reference','Comparison','Latency'};
out_data = headers;

% select comparisons of different onset times (time b minus time a; [a,b])
all_labs = {'MIDI_start','MIDI_sent','AUD'};
all_comparisons = {[2,3]}; % only comparisons of interest
%all_comparisons = {[2,3],[1,3]}; % all comparisons

%% Perform main loop to extract onsets
for i = 1:num_files
    
    % get file name
    cur_file = all_filenames{i};
    
    % get file information (separated by '_')
    cur_ind = strfind(cur_file,'_');
    cur_device = cur_file(1:cur_ind(1)-1);
    cur_sound = cur_file(cur_ind(1)+1:cur_ind(2)-1);
    
    % get onset times
    cur_onsets  = SMIDIT_soundmod_onsets(fullfile(data_dir,cur_file),params);
    
    % save data for each comparison
    for n = 1:length(all_comparisons)
        
        % get comparison data
        cur_comp = all_comparisons{n};
        cur_latencies = cur_onsets(:,cur_comp(2))-cur_onsets(:,cur_comp(1));
        num_onsets = length(cur_latencies);
        out_data_temp = cell(num_onsets,length(headers));
                
        % add label columns
        cur_ID_temp = 1:num_onsets;
        out_data_temp(:,1) = num2cell(cur_ID_temp(:));
        out_data_temp(:,2) = repmat({cur_device},[num_onsets,1]);
        out_data_temp(:,3) = repmat({cur_sound},[num_onsets,1]);        
        out_data_temp(:,4) = repmat(all_labs(cur_comp(1)),[num_onsets,1]);
        out_data_temp(:,5) = repmat(all_labs(cur_comp(2)),[num_onsets,1]);
        out_data_temp(:,6) = num2cell(cur_latencies);
                
        % add to data cell
        out_data = [out_data;out_data_temp];
                
    end
    
end

% save data to excel (if the data file is too large, cellwrite.m is 
% recommended: https://nl.mathworks.com/matlabcentral/fileexchange/7363-cellwrite
xlswrite(sprintf('SMIDIBT_soundmod_benchmark_%s.xls',date),out_data);