function SMIDIBT_route_latencies(data_dir)
% SMIDIBT_Route_latencies(data_dir)
%   Example script for extracting MIDI onset and offset times using the
%   Schultz MIDI Benchamrking Toolbox to get the latencies of all files 
%   within a folder
%
%   INPUT:
%   data_dir        The name of a folder containing the ".wav" files for
%                   the Send MIDI triggers (ending in "_send") and also
%                   contains the complementary read MIDI files (ending in 
%                   "_read") to extract the Read MIDI trigger onsets. 
%                   The default is the folder "ArduinoRoute_data" provided 
%                   in the downloaded file.
%
%   Dependencies can be found in the "MATLAB" folder of the SMIDIB Toolbox
%   hosted here: www.band-lab.com/smidib_toolbox
%
%   2017-09-28 ben.schultz@maastrichtuniversity.nl
%   Copyright (c) 2017, Benjamin Schultz, Maastricht University.

%   This script is described in more detail in the publication:
%   Schultz, B. G. (submitted).The Schultz MIDI Benchmarking Toolbox for
%   MIDI interfaces, percussion pads, and sound cards. Behavior Research 
%   Methods.
%
%   The SMIDIB Toolbox is distributed in the hope that it will be useful but
%   WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%   General Public License for more details.
% 
%   You should have received a copy of the GNU General Public License
%   along with the SMIDIB Toolbox; if not, write to the Free Software
%   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
%   02110-1301 USA
% 
%   See the file "COPYING" for the text of the license.

%% Preset necessary information
% set directory for .wav files
if ~exist('data_dir','var')
    data_dir = fullfile(cd,'route_data');
end

% set the parameters (if not the default)
params.plot_dir = fullfile(cd,'route_plots');
params.out_dir = data_dir;

% find all .wav files
all_filenames = dir(fullfile(data_dir,'*.wav'));
all_filenames = {all_filenames.name};

% select only the trigs
file_ind = strfind(all_filenames,'_send');
all_filenames = all_filenames(cellfun(@isempty,file_ind)==0);
num_files = length(all_filenames);

% preset output headers
headers = {'ID','Device','Program','Trial','Interval',...
    'Send','Transit',...
    'Read','Total'};
out_data = headers;

% select comparisons of different onset times (time b minus time a; [a,b])
all_comparisons = {[1,2],[2,3],[3,4],[1,4]}; % only comparisons of interest

%% Perform main loop to extract onsets
for i = 1:num_files
    
    % get file name
    cur_file = all_filenames{i};
    
    % get file information (separated by '_')
    cur_ind = strfind(cur_file,'_');
    cur_device = cur_file(1:cur_ind(1)-1); % hardware routing through
    cur_program = cur_file(cur_ind(1)+1:cur_ind(2)-1); % PC software for parsing MIDI
    cur_trial = cur_file(cur_ind(3)+1:cur_ind(4)-1); % Trial number
    cur_interval = cur_file(cur_ind(4)+1:cur_ind(5)-1); % interval between MIDI messages
    
    % get onset times
    cur_onsets  = SMIDIT_ArduinoRoute_onsets(fullfile(data_dir,cur_file),params);   
    num_onsets = size(cur_onsets,1);
    out_data_temp = cell(num_onsets,length(headers));
    
    % add label columns
    cur_ID_temp = 1:num_onsets;
    out_data_temp(:,1) = num2cell(cur_ID_temp(:));
    out_data_temp(:,2) = repmat({cur_device},[num_onsets,1]);
    out_data_temp(:,3) = repmat({cur_program},[num_onsets,1]);
    out_data_temp(:,4) = repmat({cur_trial},[num_onsets,1]);
    out_data_temp(:,5) = repmat({cur_interval},[num_onsets,1]);
        
    %% get comparison data AND save data for each comparison
    for n = 1:length(all_comparisons)
        cur_comp = all_comparisons{n};
        cur_latencies = cur_onsets(:,cur_comp(2))-cur_onsets(:,cur_comp(1));    
        out_data_temp(:,5+n)=num2cell(cur_latencies);
    end
    
    % add to data cell
    out_data = [out_data;out_data_temp];
    
    
end

% save data to excel (if the data file is too large, cellwrite.m is 
% recommended: https://nl.mathworks.com/matlabcentral/fileexchange/7363-cellwrite
xlswrite(sprintf('SMIDIBT_RouteTest_%s.xls',date),out_data);